;;; hello-name.lisp - A simple croatoan "Hello World" + name input example
;;;
;;; Prerequisites:
;;;   1. ncurses must be installed on your system
;;;      macOS: brew install ncurses (usually pre-installed)
;;;      Linux: apt-get install libncurses-dev
;;;   2. Load croatoan via Quicklisp:
;;;      (ql:quickload :croatoan)
;;;
;;; To run: Load this file and call (hello-name)
;;; Note: Run from a real terminal, not from SLIME/Emacs REPL
;;;       (ncurses doesn't work properly in SLIME)

(defpackage :hello-name
  (:use :cl :croatoan)
  (:export #:hello-name))

(in-package :hello-name)

(defun hello-name ()
  "Display 'Hello World' and prompt for the user's name."
  (with-screen (scr :input-echoing t
                    :input-buffering nil
                    :cursor-visible t
                    :enable-colors t)
    ;; Clear and set up
    (clear scr)

    ;; Display "Hello, World!" at top
    (move scr 2 5)
    (setf (color-pair scr) '(:cyan :black))
    (setf (attributes scr) '(:bold))
    (add-string scr "Hello, World!")

    ;; Prompt for name
    (setf (attributes scr) nil)
    (setf (color-pair scr) '(:white :black))
    (move scr 4 5)
    (add-string scr "Enter your name: ")
    (refresh scr)

    ;; Read the name character by character
    (let ((name (make-array 0 :element-type 'character
                              :adjustable t
                              :fill-pointer 0)))
      (event-case (scr event)
        ;; Enter key - finish input
        (#\Newline
         (return-from event-case))
        (#\Return
         (return-from event-case))

        ;; Backspace handling
        (#\Backspace
         (when (> (length name) 0)
           (vector-pop name)
           ;; Move cursor back and erase character
           (move scr 4 (+ 22 (length name)))
           (add-char scr #\Space)
           (move scr 4 (+ 22 (length name))))
         (refresh scr))

        ;; Delete key (also backspace on some terminals)
        (#\Rubout
         (when (> (length name) 0)
           (vector-pop name)
           (move scr 4 (+ 22 (length name)))
           (add-char scr #\Space)
           (move scr 4 (+ 22 (length name))))
         (refresh scr))

        ;; Regular character input
        (otherwise
         (when (and (characterp event)
                    (graphic-char-p event))
           (vector-push-extend event name)
           (add-char scr event)
           (refresh scr))))

      ;; Display greeting
      (setf (color-pair scr) '(:green :black))
      (setf (attributes scr) '(:bold))
      (move scr 6 5)
      (add-string scr (format nil "Nice to meet you, ~A!" name))

      (setf (attributes scr) nil)
      (setf (color-pair scr) '(:white :black))
      (move scr 8 5)
      (add-string scr "Press any key to exit...")
      (refresh scr)

      ;; Wait for a keypress before exiting
      (get-char scr)

      ;; Return the name
      name)))